(async function(PLUGIN_ID) {
    'use strict';
    const config = kintone.plugin.app.getConfig(PLUGIN_ID);
    if (!config) return;
	//
    const dateFields = config.dates ? config.dates.split(',').map(field => field.trim()) : [];
    const textColors = config.textColors ? config.textColors.split(',').map(color => color.trim()) : [];
    const backColors = config.backColors ? config.backColors.split(',').map(color => color.trim()) : [];
    const titleField = config.title;
	const suffixes = (config.suffixes || '').split(',');
	let hoyoDiv = '';
	let eventCache = {};
	let calendar;

    // 一覧表示イベントでカレンダーを初期化
	const events = ['app.record.index.show','mobile.app.record.index.show'];
    kintone.events.on(events, (event) => {
        if (event.viewId !== parseInt(config.listId)) return;
		hoyoDiv = document.getElementById(config.divId);
		if (!hoyoDiv) return;

        if (hoyoDiv.innerHTML === '') {
            initializeCalendar();
        }
    });

    const initializeCalendar = () => {
		const hoyoDiv = document.getElementById(config.divId);
		if (!hoyoDiv) return;
	
		// 「データ更新」ボタンを自動生成
		if (!document.getElementById('refresh-data')) {
			const refreshButton = document.createElement('button');
			refreshButton.id = 'refresh-data';
			refreshButton.textContent = 'データ更新';
			refreshButton.style.margin = '6px';
			refreshButton.style.color = 'white';
			refreshButton.style.backgroundColor = '#3399dd';
			refreshButton.style.border = 'none';
			refreshButton.style.borderRadius = '6px';
			refreshButton.style.padding = '2px 8px';
			hoyoDiv.parentElement.insertBefore(refreshButton, hoyoDiv);
		}
	
		// 「データ更新」ボタンのイベントリスナーを設定
		document.getElementById('refresh-data').addEventListener('click', () => {
			eventCache = {}; // キャッシュをクリア
			if (calendar) {
				calendar.refetchEvents(); // 最新データを取得
			}
		});

		// カレンダーの初期化
        calendar = new FullCalendar.Calendar(hoyoDiv, {
            initialView: kintone.mobile.app.getId() ? 'timeGridWeek' : 'dayGridMonth',
            locale: 'ja',
			headerToolbar: {
				left: 'prev,next today',
				center: 'title',
				right: 'dayGridMonth,timeGridWeek,timeGridDay,listWeek'
			},
			buttonText: {
				today: '今日',
				month: '月',
				week: '週',
				day: '日',
				list: 'リスト'
			},
            events: async (info, successCallback, failureCallback) => {
                const startDate = info.startStr;
                const endDate = info.endStr;
                const events = await fetchEvents(startDate, endDate);
                successCallback(events);
            },
            datesSet: async (info) => {
                const startDate = info.start.toISOString().split('T')[0];
                const endDate = info.end.toISOString().split('T')[0];
				const cacheKey = `${startDate}_${endDate}`;
				// 既存のイベントソースをすべて削除（レコードが２件ずつ表示されるのを防止）
				calendar.getEventSources().forEach(eventSource => eventSource.remove());
				// キャッシュにデータがある場合は、それを使用
				if (eventCache[cacheKey]) {
					calendar.removeAllEvents();
					calendar.addEventSource(eventCache[cacheKey]);
					return;
				}
				// キャッシュがない場合はAPIリクエストを行う
				const events = await fetchEvents(startDate, endDate);
				calendar.removeAllEvents();
				calendar.addEventSource(events);
				// キャッシュに保存
				eventCache[cacheKey] = events;
            },
			eventClick: (info) => {
				info.jsEvent.preventDefault();
				const appId = kintone.mobile.app.getId() || kintone.app.getId();
				const recordId = info.event.extendedProps.recordId;
			
				// モバイル環境の判定
				const isMobile = /\/k\/m\//.test(location.pathname);
			
				// モバイル版とPC版でURLを切り替え
				const recordUrl = isMobile
					? `${location.origin}/k/m/${appId}/show?record=${recordId}`
					: `${location.origin}/k/${appId}/show#record=${recordId}`;
				window.open(recordUrl, '_blank');
			}
        });

        calendar.render();
    };

    const fetchEvents = async (startDate, endDate) => {
		if (dateFields.length === 0) {
			// 日付フィールドがひとつも指定されていない場合は空配列を返す。
			return [];
		}
        const query = dateFields.map(field => {
            return `(${field} >= "${startDate}" and ${field} <= "${endDate}")`;
        }).join(' or ');

        try {
            const resp = await kintone.api(kintone.api.url('/k/v1/records', true), 'GET', {
                app: kintone.app.getId() || kintone.mobile.app.getId(),
                query: query
            });

            const events = [];
            for (const record of resp.records) {
                const title = record[titleField].value;

                dateFields.forEach((dateField, index) => {
                    if (record[dateField] && record[dateField].value) {
                        const bgColor = backColors[index] || '#ffffff';
						const dateValue = record[dateField].value;
						const isDatetimeField = dateValue.includes('T');
						// 時刻の取得（日時フィールドの場合のみ）
						const dateObj = new Date(dateValue);
						const timeString = isDatetimeField ? `${dateObj.getHours()}:${dateObj.getMinutes().toString().padStart(2, '0')}` : '';
						// タイトルの後ろに表示する文字列を取得（不足している場合は空文字）
						const suffix = suffixes[index] || '';
						// タイトルの作成（時刻を表示）
						const eventTitle = isDatetimeField ? `${timeString} ${title} ${suffix}` : `${title} ${suffix}`;
                        events.push({
                            title: eventTitle,
                            start: dateValue,
                            url: `${location.origin}/k/${kintone.app.getId()}/show#record=${record['$id'].value}`,
                            textColor: textColors[index] || '#000000',
                            backgroundColor: bgColor,
                            borderColor: bgColor,
							allDay: true,
							extendedProps: {
								recordId: record['$id'].value
							}
                        });
                    }
                });
            }
            return events;
        } catch (error) {
            alert('エラー:', error.message);
            return [];
        }
    };
})(kintone.$PLUGIN_ID);
